#include <nds.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include <nds/arm9/console.h>


#include "smm.gfx.h"      // splash image
#include "title.gfx.h"    // title image
#include "keyboard.gfx.h" // keyboard image
#include "bg.gfx.h"       // text background image
#include "font.gfx.h"     // font image
#include "cursor.gfx.h"   // cursor image
#include "dialogbottom.gfx.h" // bottom dialog box image
#include "dialogtop.gfx.h"

#include "gba_nds_fat/gba_nds_fat.h"

#define PEN_DOWN (~IPC->buttons & (1 << 6))

#define F_1	0x80
#define F_2	0x81
#define F_3	0x82
#define F_4	0x83
#define F_5	0x84
#define F_6	0x85
#define F_7	0x86
#define F_8	0x87
#define F_9	0x88
#define F10	0x89
#define F11	0x8a
#define F12	0x8b

#define EXT	0x8c // Exit

#define HOM	0xf // Home
#define PGU	0x3 // Page Up
#define PGD	0x5 // Page Down
#define END	0x10 // End

#define TAB	0x9

#define ESC	0x1b // Escape
#define BSP	0x8 // Backspace
#define CAP	0x2 // Caps
#define RET	'\n' // Enter
#define SHF	0x4 // Shift
#define	CTL	0x1d // Ctrl
#define SPC	0x20 // Space
#define ALT	0x38 // Alt
#define DSB	0x4a // DS
#define SCN	0x46 // Screen

#define CRU	0xb // Cursor up
#define CRD	0xc // Cursor down
#define CRL	0xd // Cursor Left
#define CRR	0xe // Cursor Right

#define INS	0x52 // Insert
#define DEL	0x53 // Delete

#define MAXLINE 256
#define MAXCHAR 8448

typedef struct {
  bool shift;
  bool caps;
  bool ctrl;
  bool alt;
} Keyboard;
Keyboard keyboard;

typedef struct {
  char sbuffer[MAXLINE][34]; // screen buffer
  int line_count;            // number of lines
  int line_counts[MAXLINE];  // number of characters per line
  int last_space[MAXLINE];   // last space in a row
} ScreenBuffer;
ScreenBuffer sbuffer;

typedef struct {
  int next_row;
  int next_col;
  int edit_row;
  int edit_col;
  int start_line;            // line to start printing on
  int print_rows;            // max number of rows to print
  int print_cols;
  int row_offset;
  int col_offset;
} Console;
Console console;

uint16 * font = (uint16 *)SCREEN_BASE_BLOCK(31);
uint16 * dialog = (uint16 *)SCREEN_BASE_BLOCK(27);
uint16 * td = (uint16 *) SCREEN_BASE_BLOCK(25);
uint16 * bd = (uint16 *) SCREEN_BASE_BLOCK_SUB(31);

int row_offset = 9;

int key_count = 0;		// count down for repeats

char save_file[28];
int start_file = 0;
int select_file = 0;

bool moved = FALSE;
char tmp[MAXCHAR];

static unsigned int lasttilex=0, lasttiley=0;
static char lastkey = 0x0;

//a global copy of sprite attribute memory
SpriteEntry sprites[128];

extern void insert_buffer(char c);

//-------------------------------------------------------------
void Vblank() {
//-------------------------------------------------------------

}

//-------------------------------------------------------------
void waitForVBlank() {
//-------------------------------------------------------------
	while(!(REG_DISPSTAT & DISP_IN_VBLANK));
	while((REG_DISPSTAT & DISP_IN_VBLANK));
}


//*****************************
// SPRITE FUNCTIONS
//*****************************
//turn off all the sprites
void initSprites(void) {

  for(int i = 0; i < 128; i++) {
    sprites[i].attribute[0] = ATTR0_DISABLED;
    sprites[i].attribute[1] = 0;
    sprites[i].attribute[2] = 0;
    sprites[i].attribute[3] = 0;
  }
}

//copy our sprite to object attribute memory
void updateOAM(void) {
  DC_FlushAll();
  dmaCopy(sprites, OAM, 128 * sizeof(SpriteEntry));
}

void moveCursor() {
  int x_adj;
  if(console.edit_col > console.print_cols) {
    x_adj = 1;
  } else {
    x_adj = 0;
  }

  int x = ((console.edit_col + console.col_offset - x_adj) * 8)-1;
  int y = ((console.edit_row + console.row_offset - console.start_line) * 8);

  sprites[0].attribute[0] = ATTR0_COLOR_16 | ATTR0_SQUARE | y;
  sprites[0].attribute[1] = ATTR1_SIZE_8 | x;

  updateOAM();
}


//*****************************
// ECHO FUNCTIONS
//*****************************

//-------------------------------------------------------------
void echo(char *text) {
//-------------------------------------------------------------
  waitForVBlank();

  for(int x = 0; x < (int)strlen(text); x++) {
    if(text[x] == RET) {
      console.next_row++;
      console.next_col = console.col_offset;

      if(console.next_row >= console.print_rows + console.row_offset) {
        console.next_row = console.print_rows + console.row_offset - 1;
        console.start_line++;
      }
    } else {
      font[console.next_row*32+console.next_col] = font_Map[(int)text[x]];
      console.next_col++;
      if(console.next_col >= console.print_cols + console.col_offset) {
        console.next_row++;
        console.next_col = console.col_offset;

        if(console.next_row >= console.print_rows + console.row_offset) {
          console.next_row = console.print_rows + console.row_offset - 1;
          console.start_line++;
        }
      }
    }
  }
}

//-------------------------------------------------------------
void echoc(char c) {
//-------------------------------------------------------------
  waitForVBlank();

  if(c == RET) {
    console.next_row++;
    console.next_col = console.col_offset;

    if(console.next_row >= console.print_rows + console.row_offset) {
      console.next_row = console.print_rows + console.row_offset - 1;
      console.start_line++;
    }
  } else {
    font[console.next_row*32+console.next_col] = font_Map[(int)c];
    console.next_col++;
    if(console.next_col >= console.print_cols + console.col_offset) {
      console.next_row++;
      console.next_col = console.col_offset;

      if(console.next_row >= console.print_rows + console.row_offset) {
        console.next_row = console.print_rows + console.row_offset - 1;
        console.start_line++;
      }
    }
  }
}

//-------------------------------------------------------------
void echo_clear() {
//-------------------------------------------------------------
  // clear Console
  for(int i = 0; i < 24; i++)
    for(int j = 0; j < 32; j++)
      font[i*32+j] = font_Map[0];

  console.next_row = console.row_offset;
  console.next_col = console.col_offset;
}

//-------------------------------------------------------------
void echo_clear_line() {
//-------------------------------------------------------------
  // clear line
  for(int j = 0; j < 32; j++)
    font[console.next_row*32+j] = font_Map[0];

  console.next_col = console.col_offset;
}

//-------------------------------------------------------------
void echo_clear_endofline() {
//-------------------------------------------------------------
  // clear end of line
  for(int j = console.next_col; j < 32; j++)
    font[console.next_row*32+j] = font_Map[0];

  console.next_col = console.col_offset;
}

//-------------------------------------------------------------
void echo_back() {
//-------------------------------------------------------------
  if(console.next_col == 0 && console.next_row != 0) {
    console.next_row--;
    console.next_col = console.col_offset + console.print_cols - 1;
  } else {
    console.next_col--;
  }
  font[console.next_row*32+console.next_col] = font_Map[0];
}

//-------------------------------------------------------------
void echo_goto(int x, int y) {
//-------------------------------------------------------------
  console.next_row = x + console.row_offset;
  console.next_col = y + console.col_offset;
}


//*****************************
// KEYBOARD FUNCTIONS
//*****************************

//-------------------------------------------------------------
void initKeyboard(uint16 * kb_map) {
//-------------------------------------------------------------
    dmaCopy((uint16 *)keyboard_Palette, (uint16 *)BG_PALETTE_SUB, 32);
    dmaCopy((uint16 *)keyboard_Palette_Hilight, (uint16 *)BG_PALETTE_SUB+16, 32);
    dmaCopy((uint16 *)keyboard_Tiles, (uint16 *)CHAR_BASE_BLOCK_SUB(1), 14464);

    int shift_offset = 0;

    int blank_tile = 14464;

    if((keyboard.shift && !keyboard.caps) ||
       (!keyboard.shift && keyboard.caps)) {
      shift_offset = 512;
    }

    for(int i = 0; i < row_offset*32; i++) {
      kb_map[i] = blank_tile;
    }
    for(int j = 0; j < (14)*32; j++) {
      kb_map[j+(row_offset*32)] = keyboard_Map[j+shift_offset];
    }
    for(int k = 0; k < (24-14-row_offset)*32; k++) {
      kb_map[k+((row_offset+14)*32)] = blank_tile;
    }
}

//-------------------------------------------------------------
void setTile(uint16 *map, int x, int y, int pal) {
//-------------------------------------------------------------
  char c;
  int x2, y2;

  c = keyboard_Hit[((y-row_offset)*32)+x];

  if(!c) return;

  map[(y*32)+x] &= ~(1 << 12);
  map[(y*32)+x] |= (pal << 12);

  x2 = x; y2 = y;
  while(keyboard_Hit[((y2-row_offset)*32)+x2]==c) {

    map[(y2*32)+x2] &= ~(1 << 12);
    map[(y2*32)+x2] |= (pal << 12);

    x2 = x;
    while(keyboard_Hit[((y2-row_offset)*32)+x2]==c) {
      map[(y2*32)+x2] &= ~(1 << 12);
      map[(y2*32)+x2] |= (pal << 12);
      x2++;
    }
    x2 = x;
    while(keyboard_Hit[((y2-row_offset)*32)+x2]==c) {
      map[(y2*32)+x2] &= ~(1 << 12);
      map[(y2*32)+x2] |= (pal << 12);
      x2--;
    }

    x2 = x;
    y2++;
  }

  x2 = x; y2 = y;
  while(keyboard_Hit[((y2-row_offset)*32)+x2]==c) {

    map[(y2*32)+x2] &= ~(1 << 12);
    map[(y2*32)+x2] |= (pal << 12);

    x2 = x;
    while(keyboard_Hit[((y2-row_offset)*32)+x2]==c) {
      map[(y2*32)+x2] &= ~(1 << 12);
      map[(y2*32)+x2] |= (pal << 12);
      x2++;
    }
    x2 = x;
    while(keyboard_Hit[((y2-row_offset)*32)+x2]==c) {
      map[(y2*32)+x2] &= ~(1 << 12);
      map[(y2*32)+x2] |= (pal << 12);
      x2--;
    }

    x2 = x;
    y2--;
  }
}


//*****************************
// TOUCH FUNCTIONS
//*****************************

//-------------------------------------------------------------
bool didJump() {
//-------------------------------------------------------------
  int z1, z2;
  bool jump;

  z1 = IPC->touchZ1;
  z2 = IPC->touchZ2;

  if(z1!=0 && z2!=0) {
    jump = FALSE;
  } else {
    jump = TRUE;
  }

  return jump;
}


//*****************************
// DIALOG BOX FUNCTIONS
//*****************************

//-------------------------------------------------------------
void dialog_conf() {
//-------------------------------------------------------------
  dmaCopy((uint16 *)dialogbottom_Tiles, (uint16 *)CHAR_BASE_BLOCK_SUB(0), 736);
  int blank_tile = 167;
  int bd_tile = 0;
  for(int y = 0; y < 24; y++) {
    for(int x = 0; x < 32; x++) {
      if(x < 4 || x > 27 || y > 5) {
        bd[y*32+x] = blank_tile;
      } else {
        bd[y*32+x] = dialogbottom_Map[bd_tile++];
      }
    }
  }
}

//-------------------------------------------------------------
void dialog_conf_clear() {
//-------------------------------------------------------------
  dmaCopy((uint16 *)dialogbottom_Tiles, (uint16 *)CHAR_BASE_BLOCK_SUB(0), 736);
  int blank_tile = 167;
  for(int y = 0; y < 24; y++) {
    for(int x = 0; x < 32; x++) {
      bd[y*32+x] = blank_tile;
    }
  }
}

//-------------------------------------------------------------
int dialog_read() {
//-------------------------------------------------------------
  touchPosition touch;

  int response;
  response = 0;

  scanKeys();
  touch = touchReadXY();

  if(keysDown() & KEY_A) {
    response = 1;
  } else if(keysDown() & KEY_B) {
    response = 2;
  } else {
    int x, y;
    bool jump;

    x = touch.px;
    y = touch.py;

    jump = didJump();

    if(!jump) {
      if(x >= 72 && x <= 104 &&
         y >= 16 && y <= 32 ) {
        response = 1;
      } else if(x >= 152 && x <= 184 &&
         y >= 16 && y <= 32 ) {
        response = 2;
      }
    }
  }

  return response;

}

//-------------------------------------------------------------
void dialog_alert(char * text) {
//-------------------------------------------------------------
  dmaCopy((uint16 *)dialogtop_Tiles, (uint16 *)CHAR_BASE_BLOCK(3), 224);
  int blank_tile = 287;
  int td_tile = 0;
  for(int y = 0; y < 24; y++) {
    for(int x = 0; x < 32; x++) {
      if(x < 4 || x > 27 || y < 13) {
        td[y*32+x] = blank_tile;
      } else {
        td[y*32+x] = dialogtop_Map[td_tile++];
      }
    }
  }

  int line = 0;
  int letter = 0;
  for(int x = 0; x < (int)strlen(text); x++) {
    if(text[x] == RET) {
      line++;
      letter = 0;
    } else {
      dialog[(14+line)*32+5+letter] = font_Map[(int)text[x]];
      letter++;
    }
  }
}

//-------------------------------------------------------------
void dialog_alert_clear() {
//-------------------------------------------------------------
  dmaCopy((uint16 *)dialogtop_Tiles, (uint16 *)CHAR_BASE_BLOCK(3), 224);
  int blank_tile = 167;
  for(int y = 0; y < 24; y++) {
    for(int x = 0; x < 32; x++) {
      dialog[y*32+x] = font_Map[(int)SPC];
      td[y*32+x] = blank_tile;
    }
  }
}

//-------------------------------------------------------------
bool dialog_box(char * text) {
//-------------------------------------------------------------
  bool response = FALSE;
  int input = 0;

  dialog_alert(text);
  dialog_conf();

  while(input == 0) {

    input = dialog_read();

    if(input == 1) {
      response = TRUE;
    } else if(input == 2) {
      response = FALSE;
    }
  }

  dialog_conf_clear();
  dialog_alert_clear();

  return response;
}


//*****************************
// KEYBOARD INPUT FUNCTIONS
//*****************************

//-------------------------------------------------------------
char getChar(int x, int y, uint16 * kb_map) {
//-------------------------------------------------------------
  unsigned int tilex, tiley;
  char c;

  tilex = x/8;
  tiley = y/8;

  c = (char)NULL;

  if(tilex >= 2 && tilex <= 29 &&
     (signed int)tiley <= 12+row_offset && (signed int)tiley >= 1+row_offset) {

    if((keyboard.shift && !keyboard.caps) ||
       (!keyboard.shift && keyboard.caps)) {
      c = keyboard_Hit_Shift[tilex+((tiley-row_offset)*32)];
    } else {
      c = keyboard_Hit[tilex+((tiley-row_offset)*32)];
    }

    setTile(kb_map, lasttilex, lasttiley, 0);
    setTile(kb_map, tilex, tiley, 1);
    lastkey = c; lasttilex = tilex; lasttiley = tiley;
  }

  return c;
}

//-------------------------------------------------------------
void execCaps() {
//-------------------------------------------------------------
  keyboard.caps = !keyboard.caps;
}

//-------------------------------------------------------------
void execShift() {
//-------------------------------------------------------------
  keyboard.shift = !keyboard.shift;
}

//-------------------------------------------------------------
void execBackspace() {
//-------------------------------------------------------------
  if(sbuffer.line_counts[sbuffer.line_count] > 0) {
    sbuffer.sbuffer[sbuffer.line_count][--sbuffer.line_counts[sbuffer.line_count]] = (char)NULL;

    if(sbuffer.line_count > 0 &&
       sbuffer.last_space[sbuffer.line_count-1] > 0 &&
       sbuffer.line_counts[sbuffer.line_count] < (console.print_cols + 1) - sbuffer.last_space[sbuffer.line_count-1] &&
       sbuffer.sbuffer[sbuffer.line_count-1][sbuffer.line_counts[sbuffer.line_count-1]-1] != RET) {
      // unword wrap
      int old_word = sbuffer.line_counts[sbuffer.line_count];
      for(int x = 0; x < old_word; x++) {
        sbuffer.sbuffer[sbuffer.line_count-1][sbuffer.line_counts[sbuffer.line_count-1]++] = sbuffer.sbuffer[sbuffer.line_count][x];
        sbuffer.sbuffer[sbuffer.line_count][x] = (char)NULL;
        sbuffer.line_counts[sbuffer.line_count]--;
      }
      sbuffer.line_count--;
    }
  } else if(sbuffer.line_count > 0) {
    sbuffer.line_count--;
    sbuffer.sbuffer[sbuffer.line_count][--sbuffer.line_counts[sbuffer.line_count]] = (char)NULL;
    if(console.start_line > 0) {
      console.start_line--;
    }
  }

  if(sbuffer.line_counts[sbuffer.line_count] < sbuffer.last_space[sbuffer.line_count]) {
    sbuffer.last_space[sbuffer.line_count] = 0;
    for(int y = sbuffer.line_counts[sbuffer.line_count] - 1; y >= 0; y--) {
      if(sbuffer.sbuffer[sbuffer.line_count][y] == SPC) {
        sbuffer.last_space[sbuffer.line_count] = y+1;
        break;
      }
    }
  }

}

//-------------------------------------------------------------
void execReturn() {
//-------------------------------------------------------------
  if(sbuffer.line_count < MAXLINE) {

    sbuffer.sbuffer[sbuffer.line_count][sbuffer.line_counts[sbuffer.line_count]++] = RET;
    sbuffer.line_counts[++sbuffer.line_count] = 0;

    if(sbuffer.line_count - console.print_rows >= console.start_line &&
       console.edit_row == sbuffer.line_count-1) {
      console.start_line++;
    }

  }

}

//-------------------------------------------------------------
void execChar(char c) {
//-------------------------------------------------------------
  if(c >= 32 && c <= 126 &&
     sbuffer.line_count < MAXLINE) {
    // unshift
    if(keyboard.shift) keyboard.shift = FALSE;

    // if character extends past edge of line, wrap it to next line
    if(sbuffer.line_counts[sbuffer.line_count] >= console.print_cols &&
       (c != SPC || (c == SPC && sbuffer.line_counts[sbuffer.line_count] >= console.print_cols+1))) {

      if(sbuffer.last_space[sbuffer.line_count] < console.print_cols && c != SPC &&
         sbuffer.last_space[sbuffer.line_count] > 0) {
        sbuffer.line_counts[++sbuffer.line_count] = 0;

        for(int x = sbuffer.last_space[sbuffer.line_count-1]; x < console.print_cols; x++) {
          sbuffer.sbuffer[sbuffer.line_count][sbuffer.line_counts[sbuffer.line_count]++] = sbuffer.sbuffer[sbuffer.line_count-1][x];

          sbuffer.sbuffer[sbuffer.line_count-1][x] = (char)NULL;
          sbuffer.line_counts[sbuffer.line_count-1]--;
        }
      } else {
        sbuffer.line_counts[++sbuffer.line_count] = 0;
      }

    } else if (c == SPC) {
      sbuffer.last_space[sbuffer.line_count] = sbuffer.line_counts[sbuffer.line_count] + 1;
    }

    // add char
    sbuffer.sbuffer[sbuffer.line_count][sbuffer.line_counts[sbuffer.line_count]++] = c;

    if(sbuffer.line_count - console.print_rows >= console.start_line &&
       console.edit_row == sbuffer.line_count-1) {
      console.start_line = sbuffer.line_count - console.print_rows + 1;
    }

  }
}

//-------------------------------------------------------------
void execSpace() {
//-------------------------------------------------------------
  execChar(SPC);
}

//-------------------------------------------------------------
void execKey(char c) {
//-------------------------------------------------------------
  if(console.edit_row == sbuffer.line_count &&
     console.edit_col == sbuffer.line_counts[sbuffer.line_count]) {
    if(c == RET) {
      execReturn();
    } else if(c == BSP) {
      execBackspace();
    } else {
      execChar(c);
    }

    console.edit_row = sbuffer.line_count;
    console.edit_col = sbuffer.line_counts[sbuffer.line_count];
  } else {
    insert_buffer(c);
    if(c == BSP) {
      console.edit_col--;
      if(console.edit_col < 0) {
        if(console.edit_row > 0) {
          console.edit_row--;
          console.edit_col = sbuffer.line_counts[console.edit_row];
        } else {
          console.edit_col = 0;
        }
      }
    } else if(c == RET) {
      console.edit_row++;
      console.edit_col = 0;
    } else {
      console.edit_col++;
      if(console.edit_col > sbuffer.line_counts[console.edit_row]) {
        console.edit_row++;
        console.edit_col = 0;
      }
    }
  }
}

//*****************************
// BUFFER FUNCTIONS
//*****************************

//-------------------------------------------------------------
void print_buffer() {
//-------------------------------------------------------------
  waitForVBlank();
  int print_line = console.start_line;
  console.next_col = 0;

  for(int i = console.row_offset; i < console.print_rows + console.row_offset; i++) {

    for(int j = console.col_offset; j < console.print_cols + console.col_offset; j++) {
      if(j - console.row_offset < sbuffer.line_counts[print_line + i - console.row_offset] &&
         sbuffer.sbuffer[print_line + i - console.row_offset][j - console.col_offset] != RET) {
        font[i*32+j] = font_Map[(int)sbuffer.sbuffer[print_line + i - console.row_offset][j - console.col_offset]];
      } else {
        font[i*32+j] = font_Map[0];
      }
    }
  }
}

//-------------------------------------------------------------
void save_buffer(char* sbuff) {
//-------------------------------------------------------------
  // convert buffer array into string
  for(int i = 0; i <= sbuffer.line_count; i++) {
    strcat(sbuff, sbuffer.sbuffer[i]);
  }
  strcat(sbuff, " ");
}

//-------------------------------------------------------------
void load_buffer(char* obuff) {
//-------------------------------------------------------------
  for(int i = 0; i < (int)strlen(obuff); i++) {
    if(obuff[i] == RET) {
      execReturn();
    } else if(obuff[i] == SPC) {
      execSpace();
    } else {
      execChar(obuff[i]);
    }
  }

  if(sbuffer.line_count - console.print_rows >= console.start_line) {
    console.start_line = sbuffer.line_count - console.print_rows + 1;
  }

  console.edit_row = sbuffer.line_count;
  console.edit_col = sbuffer.line_counts[sbuffer.line_count];
  moveCursor();
}

//-------------------------------------------------------------
void insert_buffer(char c) {
//-------------------------------------------------------------
  // all the lines before it
  for(int i = 0; i < console.edit_row; i++) {
    strcat(tmp, sbuffer.sbuffer[i]);
  }

  // all the characters before it in that line
  for(int j = 0; j < console.edit_col; j++) {
    tmp[strlen(tmp)] = sbuffer.sbuffer[console.edit_row][j];
  }

  if(c == BSP) {
    // backspace
    tmp[strlen(tmp)-1] = (char)NULL;
  } else {
    // insert the character
    tmp[strlen(tmp)] = c;
  }

  // all the rest of the characters on that line
  for(int j = console.edit_col; j < sbuffer.line_counts[console.edit_row]; j++) {
    tmp[strlen(tmp)] = sbuffer.sbuffer[console.edit_row][j];
  }

  // all the rest of the lines
  for(int i = console.edit_row + 1; i <= sbuffer.line_count; i++) {
    strcat(tmp, sbuffer.sbuffer[i]);
  }

  for(int i = 0; i <= sbuffer.line_count; i++) {
    for(int j = 0; j < sbuffer.line_counts[i]; j++) {
      // clear screen buffer
      sbuffer.sbuffer[i][j] = (char)NULL;
    }

    // clear count arrays
    sbuffer.line_counts[i] = 0;
    sbuffer.last_space[i] = 0;
  }

  // clear ints
  sbuffer.line_count = 0;
  console.start_line = 0;

  // process new buffer
  for(int i = 0; i < (int)strlen(tmp); i++) {
    if(tmp[i] == RET) {
      execReturn();
    } else if(tmp[i] == SPC) {
      execSpace();
    } else {
      execChar(tmp[i]);
    }
  }

  // clear tmp buffer
  for(int i = (int)strlen(tmp); i >= 0; i--) {
    tmp[i] = (char)NULL;
  }

}

//-------------------------------------------------------------
void clear() {
//-------------------------------------------------------------
  echo_clear();

  // clear arrays
  for(int i = 0; i <= sbuffer.line_count; i++) {
    for(int j = 0; j < sbuffer.line_counts[i]; j++) {
      // clear screen buffer
      sbuffer.sbuffer[i][j] = (char)NULL;
    }

    // clear count arrays
    sbuffer.line_counts[i] = 0;
    sbuffer.last_space[i] = 0;
  }

  // clear ints
  sbuffer.line_count = 0;
  console.start_line = 0;
  console.edit_row = 0;
  console.edit_col = 0;

  moveCursor();

  // clear save file name
  for(int h = strlen(save_file); h >= 0; h--) {
    save_file[h] = (char)NULL;
  }
}


//*****************************
// FILE FUNCTIONS
//*****************************

//-------------------------------------------------------------
void file_cd(char* dir) {
//-------------------------------------------------------------
  FAT_chdir(dir);
}

//-------------------------------------------------------------
u32 file_save(char* filename, char* sbuff) {
//-------------------------------------------------------------
  u32 written;
  FAT_FILE* pFile;
  pFile = FAT_fopen (filename, "w");
  written = FAT_fwrite (sbuff , 1 , strlen(sbuff) , pFile);
  FAT_fclose (pFile);
  return written;
}

//-------------------------------------------------------------
char* file_open(char* filename) {
//-------------------------------------------------------------
  FAT_FILE * rFile;
  u32 lSize;
  char* rbuffer;

  rFile = FAT_fopen (filename, "r");
  if (rFile==NULL) return "NULL";

  // obtain file size.
  lSize = FAT_GetFileSize();

  // allocate memory to contain the whole file.
  rbuffer = (char*) malloc (lSize);
  // if (rbuffer == NULL) exit (2);

  // copy the file into the buffer.
  FAT_fread (rbuffer,1,lSize,rFile);

  FAT_fclose (rFile);

  return rbuffer;
}

//-------------------------------------------------------------
bool file_del(char* filename) {
//-------------------------------------------------------------
  bool deleted;
  deleted = FAT_remove(filename);
  return deleted;
}


char fn[128][20];
//-------------------------------------------------------------
void sortfn(int num) {
//-------------------------------------------------------------
  int i, eff_size, maxpos = 0;
  char tmp[20];

  for(eff_size = num; eff_size > 1; eff_size--) {
    maxpos = 0;
    for(i = 0; i < eff_size; i++) {
      if(strcmp(fn[i], fn[maxpos]) > 0) {
        maxpos = i;
      }
    }

    strcpy(tmp, fn[maxpos]);
    strcpy(fn[maxpos], fn[eff_size-1]);
    strcpy(fn[eff_size - 1], tmp);

  }
}

//-------------------------------------------------------------
int get_dir() {
//-------------------------------------------------------------
  int type;
  int i = 0;
  char tmp[20];
  
  /*
  type = FAT_FindFirstFileLFN(tmp);
  while (type > 0) {
    if(type == 1) strcpy(fn[i++], tmp); 
    type = FAT_FindNextFileLFN(tmp);
  }
  */
  
  type = FAT_FindFirstFile(tmp);
  while (type > 0) {
    if(type == 1) strcpy(fn[i++], tmp);
    type = FAT_FindNextFile(tmp);
  }
  
  sortfn(i);

  return i;
}


//*****************************
// TXTWRITER MAIN PROGRAM
//*****************************

//-------------------------------------------------------------
int main(void) {
//-------------------------------------------------------------

  powerON(POWER_ALL);

  irqInit();
  irqSet(IRQ_VBLANK, Vblank);
  irqEnable(IRQ_VBLANK);

  //*****************************
  // VIDEO SET-UP
  //*****************************

    videoSetMode(MODE_0_2D |
                 DISPLAY_SPR_ACTIVE |
                 DISPLAY_BG0_ACTIVE |
                 DISPLAY_BG1_ACTIVE |
                 DISPLAY_BG2_ACTIVE |
                 DISPLAY_BG3_ACTIVE |
                 DISPLAY_SPR_1D);
    videoSetModeSub(MODE_0_2D | DISPLAY_BG0_ACTIVE | DISPLAY_BG1_ACTIVE);
    vramSetMainBanks(VRAM_A_MAIN_BG, VRAM_B_MAIN_SPRITE,
       VRAM_C_SUB_BG, VRAM_D_LCD);
	
	int i;
	 for (i = 0; i < (131072); i++)
	{
	BG_GFX[i] = 0;
	}
	
    // ** TEXT **
    BG2_CR = BG_COLOR_16 | BG_32x32 | (31 << SCREEN_SHIFT) | (0 << CHAR_SHIFT);

    // ** CONSOLE BACKGROUND **
    uint16 * bg = (uint16 *)SCREEN_BASE_BLOCK(29);
    BG3_CR = BG_COLOR_16 | BG_32x32 | (29 << SCREEN_SHIFT) | (1 << CHAR_SHIFT);

    // ** DIALOG TEXT **
    BG0_CR = BG_COLOR_16 | BG_32x32 | (27 << SCREEN_SHIFT) | (2 << CHAR_SHIFT);

    // ** TOP DIALOG BOX **
    BG1_CR = BG_COLOR_16 | BG_32x32 | (25 << SCREEN_SHIFT) | (3 << CHAR_SHIFT);

    // ** KEYBOARD TILES **
    uint16 * kb_map = (uint16 *) SCREEN_BASE_BLOCK_SUB(29);
    SUB_BG1_CR = BG_COLOR_16 | BG_32x32 | (29 << SCREEN_SHIFT) | (1 << CHAR_SHIFT);

    // ** BOTTOM DIALOG BOX **
    SUB_BG0_CR = BG_COLOR_16 | BG_32x32 | (31 << SCREEN_SHIFT) | (0 << CHAR_SHIFT);

    //lcdSwap();


  //*****************************
  // INPUT SET-UP
  //*****************************

    touchPosition touch;
    FAT_InitFiles();
    file_cd("/TXT");
    initSprites();


  //*****************************
  // SPLASH SCREEN
  //*****************************

    dmaCopy((uint16 *)smm_Palette, (uint16 *)BG_PALETTE_SUB, 32);
    dmaCopy((uint16 *)smm_Map, (uint16 *)kb_map, 32*24*2);
    dmaCopy((uint16 *)smm_Tiles, (uint16 *)CHAR_BASE_BLOCK_SUB(1), 2560);

    dmaCopy((uint16 *)title_Palette, (uint16 *)BG_PALETTE, 32);
    dmaCopy((uint16 *)title_Map, (uint16 *)bg, 32*24*2);
    dmaCopy((uint16 *)title_Tiles, (uint16 *)CHAR_BASE_BLOCK(1), 1632);

    do {
      scanKeys();
      waitForVBlank();
    } while(!(keysDown() & KEY_A) && !PEN_DOWN);

    do {
      scanKeys();
      waitForVBlank();
    } while((keysDown() & KEY_A) || PEN_DOWN);


  //*****************************
  // INITIATE KEYBOARD SCREEN
  //*****************************

    keyboard.shift = FALSE;
    keyboard.caps = FALSE;
    keyboard.ctrl = FALSE;
    keyboard.alt = FALSE;

    initKeyboard(kb_map);


  //*****************************
  // INITIATE TEXT SCREEN
  //*****************************
    dmaCopy((uint16 *)bg_Palette, (uint16 *)BG_PALETTE, 32);
    dmaCopy((uint16 *)bg_Map, (uint16 *)bg, 32*24*2);
    dmaCopy((uint16 *)bg_Tiles, (uint16 *)CHAR_BASE_BLOCK(1), 832);

    // ** INITIATE FONT **
    dmaCopy((uint16 *)font_Tiles, (uint16 *)CHAR_BASE_BLOCK(0), 2976);
    dmaCopy((uint16 *)font_Tiles, (uint16 *)CHAR_BASE_BLOCK(2), 2976);
    BG_PALETTE[15] = RGB15(0,0,0);

    // ** INITIATE CURSOR SPRITE **
    sprites[0].attribute[0] = ATTR0_COLOR_16 | ATTR0_SQUARE | 8;
	sprites[0].attribute[1] = ATTR1_SIZE_8 | 7;
	sprites[0].attribute[2] = ATTR2_ALPHA(1)| 0;

	dmaCopy((uint16 *)cursor_Palette, (uint16 *)SPRITE_PALETTE, 32);
	dmaCopy((uint16 *)cursor_Tiles, (uint16 *)SPRITE_GFX, 8*8*2);

	updateOAM();

  //*****************************
  // INITIATE SCREEN BUFFER
  //*****************************

    sbuffer.line_count = 0;

    console.start_line = 0;
    console.print_rows = 22;
    console.print_cols = 30;
    console.row_offset = 1;
    console.col_offset = 1;
    console.next_row = console.row_offset;
    console.next_col = console.col_offset;
    console.edit_row = 0;
    console.edit_col = 0;


  int keyboard_ctrl = 0; // 1 for save
                         // 2 for open
                         // 3 for clear
                         // 4 for help

  while(1) {

    scanKeys();
    touch = touchReadXY();

    if(keysDown() & KEY_SELECT ||
       keyboard_ctrl == 3) {

      keyboard_ctrl = 0;
      clear();

    } else if(keysDown() & KEY_START) {

      console.edit_row = 1;
      console.edit_col = 1;
      execKey('c');
      print_buffer();

    } else if(keysDown() & KEY_L ||
              keyboard_ctrl == 1){
     //*****************************
     // SAVE FILE MENU
     //*****************************

      keyboard_ctrl = 0;
      bool save_it = FALSE;
      bool cancel_it = FALSE;

      echo_clear();
      echo("SAVE MENU\n");
      echo("------------------------------");
      echo("Filename: ");
      echo(save_file);

      dialog_conf();
      int dialog_res;

      while(1) {

        dialog_res = dialog_read();

        if(dialog_res == 1) {
          save_it = TRUE;
          dialog_conf_clear();
        } else if(dialog_res == 2) {
          cancel_it = TRUE;
          dialog_conf_clear();
        } else {
         //*****************************
         // FILENAME DIALOG
         //*****************************

          int x, y;
          bool jump;
          touch = touchReadXY();

          x = touch.px;
          y = touch.py;

          jump = didJump();

          if(!jump) {
            char c;

            c = getChar(x, y, kb_map);

            if(c != (char)NULL) {

              if(c == RET) { // Return
                save_it = TRUE;
              } else if(c == BSP) { // Backspace
                if(strlen(save_file) > 0) {
                  echo_back();
                  save_file[strlen(save_file)-1] = (char)NULL;
                }
              } else if(c == CAP) { // Caps
                execCaps();
              } else if(c == SHF) { // Shift
                execShift();
              } else {
                if(c >= 32 && c <= 126) {
                  if(keyboard.shift) keyboard.shift = FALSE;
                  if(strlen(save_file) <= 27) {
                    echoc(c);
                    save_file[strlen(save_file)] = c;
                  }
                }
              }

              initKeyboard(kb_map);
              key_count = 10;
            }
          }
        }

        do {
          key_count--;
          waitForVBlank();
        } while(key_count > 0);

        waitForVBlank();

        if(save_it || cancel_it) break;

      }

      if(save_it) {
       //*****************************
       // SAVE FILE
       //*****************************
        u32 written;
        char full_save[32];
        char default_file[] = "txtwriter";
        char file_ext[] = ".txt";
        if(strlen(save_file) == 0) {
          strcpy(save_file, default_file);
        }
        strcpy(full_save, save_file);
        strcat(full_save, file_ext);

        bool overwrite = TRUE;
        int num_files;
        num_files = get_dir();
        char tmp_fn[32];
        char tmp_sn[32];
        strcpy(tmp_sn, full_save);
        for(int z = 0; z < (int)strlen(tmp_sn); z++) tmp_sn[z] = toupper(tmp_sn[z]);

        for(int l = 0; l < num_files; l++) {
          strcpy(tmp_fn, fn[l]);
          for(int z = 0; z < (int)strlen(tmp_fn); z++) tmp_fn[z] = toupper(tmp_fn[z]);
          if(strcmp(tmp_fn,tmp_sn) == 0) {
           //*****************************
           // OVERWRITE DIALOG
           //*****************************
            echo_clear();
            echo(full_save);
            echo(" already exists");

            overwrite = dialog_box("Overwrite file?");
          }
        }

        if(overwrite) {
         //*****************************
         // WRITE FILE
         //*****************************
          save_buffer(tmp);

          dialog_alert("Saving...");
          written = file_save(full_save, tmp);
          for(int x = strlen(tmp); x > 0; x--) tmp[x-1] = (char)NULL;
          dialog_alert_clear();

          if(written > 0) {
            print_buffer();
            dialog_box("File saved...");
          } else {
            print_buffer();
            dialog_box("ERROR: save failed");
          }

        } else {
          waitForVBlank();
          print_buffer();
        }
      } else {
        waitForVBlank();
        print_buffer();
      }

    } else if(keysDown() & KEY_R ||
              keyboard_ctrl == 2){
     //*****************************
     // FILE MENU
     //*****************************

      keyboard_ctrl = 0;
      int num_files = 0;
      int print_lines = 24-4-1;
      int key_repeat = 0;

      bool open_it = FALSE;
      bool cancel_it = FALSE;
      bool refresh = TRUE;
      bool move_list = TRUE;

      echo_clear();
      echo("FILE MENU\n");
      echo("------------------------------");

      num_files = get_dir();
      dialog_conf();
      int dialog_res;

      while(1) {
        if(refresh) {

          if(num_files > 0) {
            int dir_list;
            if(num_files < print_lines) {
              dir_list = num_files;
            } else {
              dir_list = print_lines;
            }

           //*****************************
           // FILE LIST
           //*****************************
            for(int i = 0; i < dir_list; i++) {
              echo_goto(2+i,0);
              if(select_file >= dir_list) select_file = dir_list-1;
              if(i + start_file <= select_file + 1 ||
                 i + start_file >= select_file -1) {
                if(i + start_file == select_file) {
                  echo(" * ");
                } else {
                  echo("   ");
                }
              }
              if(move_list) {
                echo(fn[i + start_file]);
                echo_clear_endofline();
              }
            }
            if(move_list) {
              echoc(RET);
              echo_clear_line();
            }
            move_list = FALSE;
          } else if(move_list) {
            echo_goto(2,0);
            echo_clear_line();
            move_list = FALSE;
          }
          refresh = FALSE;
        }

        scanKeys();

        if(keysDown() & KEY_UP) {
          if(start_file == select_file && start_file > 0) {
            start_file--;
            move_list = TRUE;
          }
          if(select_file > 0) select_file--;
          key_repeat = 2;
          refresh = TRUE;
        } else if(keysDown() & KEY_DOWN) {
          if(start_file + print_lines-1 == select_file && select_file < num_files-1) {
            start_file++;
            move_list = TRUE;
          }
          if(select_file < num_files-1) select_file++;
          key_repeat = 2;
          refresh = TRUE;

        } else if(keysDown() & KEY_X) {
         //*****************************
         // DELETE DIALOG
         //*****************************
          char del_file[20];
          strcpy(del_file,fn[select_file]);

          bool delete_file;
          bool deleted;

          dialog_conf_clear();
          delete_file = dialog_box("Delete file?");

          if(delete_file) {
            dialog_alert("Deleting...");
            deleted = file_del(del_file);
            dialog_alert_clear();

            if(deleted) {
              num_files = get_dir();
              dialog_box("File deleted.");

              refresh = TRUE;
              move_list = TRUE;
            } else {
              dialog_box("ERROR: delete failed.");
            }

            int pause = 0;
            while(pause < 10) {
              waitForVBlank();
              pause++;
            }
          }

          dialog_conf();

        }

        dialog_res = dialog_read();
        if(dialog_res == 1) {
          open_it = TRUE;
          dialog_conf_clear();
        } else if(dialog_res == 2) {
          cancel_it = TRUE;
          dialog_conf_clear();
        }

        if(key_repeat == 0) {
          if(keysHeld() & KEY_UP) {
            if(start_file == select_file && start_file > 0) start_file--;
            if(select_file > 0) select_file--;
            key_repeat = 0;
            refresh = TRUE;
          } else if(keysHeld() & KEY_DOWN) {
            if(start_file + print_lines-1 == select_file && select_file < num_files-1) start_file++;
            if(select_file < num_files-1) select_file++;
            key_repeat = 0;
            refresh = TRUE;
          }
        }

        if(key_repeat > 0) key_repeat--;

        waitForVBlank();

        if(open_it || cancel_it) break;

      }

      if(open_it) {
       //*****************************
       // OPEN FILE
       //*****************************
        clear();
        dialog_alert("Processing...");

        strcpy(tmp,file_open(fn[select_file]));

        for(int h = strlen(save_file); h >= 0; h--) {
          save_file[h] = (char)NULL;
        }
        strncpy(save_file, fn[select_file], strlen(fn[select_file])-4);
        load_buffer(tmp);
        for(int x = strlen(tmp); x > 0; x--) tmp[x-1] = (char)NULL;
        waitForVBlank();
        print_buffer();
        dialog_alert_clear();

      } else {
        waitForVBlank();
        echo_clear();
        print_buffer();
      }

     //*****************************
     // BUTTON CONTROLS
     //*****************************

    } else if((keysDown() & KEY_UP) ||
              (keysDown() & KEY_X)) {

      execShift();
      initKeyboard(kb_map);

    } else if((keysDown() & KEY_RIGHT) ||
              (keysDown() & KEY_A)) {

      execKey(SPC);
      print_buffer();
      moveCursor();

    } else if((keysDown() & KEY_LEFT) ||
              (keysDown() & KEY_Y)) {

      execKey(BSP);
      print_buffer();
      moveCursor();

    } else if((keysDown() & KEY_DOWN) ||
              (keysDown() & KEY_B)) {

      execKey(RET);
      print_buffer();
      moveCursor();

    } else {

     //*****************************
     // KEYBOARD PROCESSING
     //*****************************

      int x, y;
      bool jump;

      x = touch.px;
      y = touch.py;

      jump = didJump();

      if(!jump) {
        char c;

        c = getChar(x, y, kb_map);

        if(c != (char)NULL) {

          if(c == F_1) {
            keyboard_ctrl = 1;  // save
          } else if(c == F_2) {
            keyboard_ctrl = 2;  // open
          } else if(c == EXT) {
            keyboard_ctrl = 3;  // clear
          } else if(c == PGU) { // Page Up
            if(console.start_line - console.print_rows + 1 > 0) {
              console.start_line = console.start_line - console.print_rows + 1;
              console.edit_row = console.start_line + 1;
            } else {
              console.start_line = 0;
              console.edit_row = 0;
            }
            console.edit_col = 0;
          } else if(c == PGD) { // Page Down
            if(sbuffer.line_count > console.print_rows) {
              if(console.start_line + console.print_rows - 1 <= sbuffer.line_count - console.print_rows + 1) {
                console.start_line = console.start_line + console.print_rows - 1;
                console.edit_row = console.start_line + console.print_rows - 2;
              } else {
                console.start_line = sbuffer.line_count - console.print_rows + 1;
                console.edit_row = sbuffer.line_count;
              }
            } else {
              console.edit_row = sbuffer.line_count;
            }
            console.edit_col = 0;
          } else if(c == CRU || c == CRD) {
            if(c == CRU) {
              if(console.edit_row > 0) {
                console.edit_row--;
                if(console.edit_row < console.start_line) {
                   console.start_line--;
                }
              }
            } else {
              if(console.edit_row < sbuffer.line_count) {
                console.edit_row++;
                if(console.edit_row > console.start_line + console.print_rows - 1) {
                   console.start_line++;
                }
              }
            }
            if(console.edit_col >= sbuffer.line_counts[console.edit_row]) {
              if(sbuffer.sbuffer[console.edit_row][sbuffer.line_counts[console.edit_row]-1] == RET) {
                console.edit_col = sbuffer.line_counts[console.edit_row]-1;
              } else {
                console.edit_col = sbuffer.line_counts[console.edit_row];
              }
            }
          } else if(c == CRL) {
            if(console.edit_col > 0) console.edit_col--;
          } else if(c == CRR) {
            if(console.edit_col < sbuffer.line_counts[console.edit_row]) {
              if(sbuffer.sbuffer[console.edit_row][console.edit_col] != RET) {
                console.edit_col++;
              }
            }
          } else if(c == HOM) {
            console.edit_col = 0;
          } else if(c == END) {
            if(sbuffer.sbuffer[console.edit_row][sbuffer.line_counts[console.edit_row]-1] == RET) {
              console.edit_col = sbuffer.line_counts[console.edit_row]-1;
            } else {
              console.edit_col = sbuffer.line_counts[console.edit_row];
            }
          } else if(c == CAP) { // Caps
            execCaps();
          } else if(c == SHF) { // Shift
            execShift();
          } else {
            execKey(c);
          }

          waitForVBlank();

          print_buffer();
          moveCursor();
          initKeyboard(kb_map);
          key_count = 10;

        }

      }

    }

    do {
      key_count--;
      waitForVBlank();
    } while(key_count > 0);

    waitForVBlank();

  }

  return 0;

}
